/* ...............................................................

	WindowColors
	Copyright 1997-8 Steve Klingsporn <moofie@pobox.com>
	Based on WindowShade by Marco Nelissen <marcone@xs4all.nl>
	
		File:	WindowColorsPanel.cpp
	
	Contains:	Implementation of the WindowColorsPanel BView
				subclass.
	
	   Notes:	Nada.
	   
   ............................................................... */

#ifndef _WINDOW_COLORS_PANEL_H
#include "WindowColorsPanel.h"
#endif


/* ...............................................................	
	String Constants (that probably should be in resources)
   ............................................................... */

const char 	*TABBED_WINDOW_TITLE = "Example";
const char	*SET_BUTTON_TITLE = "Set Colors";
const char	*REVERT_BUTTON_TITLE = "Revert";
const char	*DEFAULTS_BUTTON_TITLE = "Defaults";
const char	*ABOUT_BUTTON_TITLE = "About...";
const char	*TITLE_TAB_LABEL_TEXT = "Title Tab";
const char	*FRAME_LABEL_TEXT = "Frame";
const char	*MODAL_FRAME_LABEL_TEXT = "Modal Frame";
const char	*WINDOW_COLORS_PANEL_NAME = "WindowColorsPanel";


/* ...............................................................	
	WindowColorsPanel Constructor
   ............................................................... */

WindowColorsPanel::WindowColorsPanel(BRect frame)
					 :BView(frame, WINDOW_COLORS_PANEL_NAME, 
					 		B_FOLLOW_ALL_SIDES, B_WILL_DRAW),
					  _activeWindowView(NULL),
					  _synced(true)
{
	for (uint8 i = 0; i < 4; i++)
		_bitmaps[i] = new BBitmap(WINDOW_BITMAP_FRAME,
								  B_COLOR_8_BIT);
}


/* ...............................................................	
	WindowColorsPanel destructor
	Deletes the bitmaps we allocated
   ............................................................... */

WindowColorsPanel::~WindowColorsPanel()
{
	for (uint8 i = 0; i < 4; i++)
		delete _bitmaps[i];
}


/* ...............................................................	
	WindowColorsPanel::GetColorPicker()
	Returns the color picker.
   ............................................................... */

ColorPicker *WindowColorsPanel::GetColorPicker()
{
	return _colorPicker;
}


/* ...............................................................	
	WindowColorsPanel::AttachedToWindow()
	Sets up the view when attached to the window.
   ............................................................... */

void WindowColorsPanel::AttachedToWindow()
{
	SetViewColor(RGB_COLOR_216);
	
	/* Create and add the model window views */
	_windowView = new WindowView(TABBED_WINDOW_FRAME,
						 	     TABBED_WINDOW_TITLE,
							     B_TITLED_WINDOW);
	AddChild(_windowView);
	
	_modalWindowView = new WindowView(MODAL_WINDOW_FRAME,
									  "", B_MODAL_WINDOW);
	AddChild(_modalWindowView);
			
	/* Create and add a groovy separator line */
	_separatorLine = new SeparatorLine(SEPARATOR_LINE_FRAME, "", 
									   B_HORIZONTAL, B_FOLLOW_NONE, 
									   B_WILL_DRAW);
	AddChild(_separatorLine);

	/* Create a container view to restrict BColorControl
	   from drawing crazily eastward */
	BView *colorControlContainer = new BView(COLOR_CONTROL_FRAME,
											 "", B_FOLLOW_ALL,
											 B_WILL_DRAW);

	_colorPicker = new ColorPicker(BPoint(0, 0));
	_colorPicker->SetValueQuietly(Color(ACTIVE_TAB_BASE_COLOR));
	colorControlContainer->AddChild(_colorPicker);
	AddChild(colorControlContainer);

	/* Start (with "OK") adding buttons (font-sensitive, but
	   somewhat intolerant (sorry, Marco)) */
	BPoint origin = BUTTON_RIGHT_EDGE;
	
	_setButton = new BButton(BRect(),
									  SET_BUTTON_TITLE,
									  SET_BUTTON_TITLE,
									  new BMessage(SET_WINDOW_COLORS));
	_setButton->SetTarget(this);
	_setButton->SetEnabled(false);
	if (be_plain_font->Size() > 14)
		_setButton->SetFontSize(14);
	_setButton->MakeDefault(true);
	_setButton->ResizeToPreferred();
	origin.x -= _setButton->Frame().Width();
	_setButton->MoveTo(origin);
	AddChild(_setButton);
	
	/* Defaults */
	origin.y += 3;
	_defaultsButton = new BButton(BRect(),
								DEFAULTS_BUTTON_TITLE,
								DEFAULTS_BUTTON_TITLE,
								new BMessage(REVERT_TO_BE_DEFAULTS));
	_defaultsButton->SetTarget(this);
	_defaultsButton->SetFlags(B_WILL_DRAW);	/* Can't steal focus */
	if (be_plain_font->Size() > 14)
		_defaultsButton->SetFontSize(14);	
	_defaultsButton->ResizeToPreferred();
	origin.x -= (BUTTON_SPACER + _defaultsButton->Frame().Width());
	_defaultsButton->MoveTo(origin);
	AddChild(_defaultsButton);
	
	/* Revert */
	_revertButton = new BButton(BRect(),
								  REVERT_BUTTON_TITLE,
								  REVERT_BUTTON_TITLE,
								  new BMessage(REVERT_WINDOW_COLORS));
	_revertButton->SetTarget(this);
	_revertButton->SetFlags(B_WILL_DRAW);
	if (be_plain_font->Size() > 14)
		_revertButton->SetFontSize(14);
	_revertButton->ResizeToPreferred();
	origin.x -= (BUTTON_SPACER + _revertButton->Frame().Width());
	_revertButton->MoveTo(origin);
	AddChild(_revertButton);
			
	/* About */
	origin.x = LEFT_ALIGNMENT_MARGIN;
	_aboutButton = new BButton(BRect(),
							   ABOUT_BUTTON_TITLE,
							   ABOUT_BUTTON_TITLE,
							   new BMessage(B_ABOUT_REQUESTED));
	_aboutButton->SetTarget(this);
	_aboutButton->SetFlags(B_WILL_DRAW);
	if (be_plain_font->Size() > 14)
		_aboutButton->SetFontSize(14);
	_aboutButton->ResizeToPreferred();
	_aboutButton->MoveTo(origin);
	AddChild(_aboutButton);
	
	/* Add some labels to the WindowViews */
	BStringView	*label = new BStringView(TITLE_TAB_LABEL_FRAME,
										 TITLE_TAB_LABEL_TEXT,
										 TITLE_TAB_LABEL_TEXT);
	
	/* Set the label attributes */
	label->SetFont(be_bold_font);
	if (be_bold_font->Size() > 14)
		label->SetFontSize(14);
	label->AddFilter(new ColumnLabelFilter());
	_windowView->AddChild(label);
	
	label = new BStringView(FRAME_LABEL_FRAME,
							FRAME_LABEL_TEXT,
							FRAME_LABEL_TEXT);
	
	/* Set the label attributes */
	label->SetFont(be_bold_font);
	if (be_bold_font->Size() > 14)
		label->SetFontSize(14);
	label->AddFilter(new ColumnLabelFilter());
	
	_windowView->AddChild(label);
	
	label = new BStringView(MODAL_FRAME_LABEL_FRAME,
							MODAL_FRAME_LABEL_TEXT,
							MODAL_FRAME_LABEL_TEXT);
	
	/* Set the label attributes */
	label->SetFont(be_bold_font);
	if (be_bold_font->Size() > 14)
		label->SetFontSize(14);
	label->AddFilter(new ColumnLabelFilter());
	_modalWindowView->AddChild(label);

	/* Fix things up a bit and get the system window colors
	   into our color model array. */
	SetActiveWindowView(_windowView);
	SetViewColor(B_TRANSPARENT_32_BIT);
	GetSystemWindowColors();
	
	/* Add keyboard shortcuts for the buttons.
	   We'll make them equivalent to the first character
	   in the button title. */
	BMessage *message = new BMessage(INVOKE_BUTTON);
	message->AddString("which", ABOUT_BUTTON_TITLE);
	Window()->AddShortcut('A', B_COMMAND_KEY,
						  message,
						  this);
						  
	message = new BMessage(INVOKE_BUTTON);
	message->AddString("which", REVERT_BUTTON_TITLE);
	Window()->AddShortcut('R', B_COMMAND_KEY,
						  message,
						  this);

	message = new BMessage(INVOKE_BUTTON);
	message->AddString("which", DEFAULTS_BUTTON_TITLE);
	Window()->AddShortcut('D', B_COMMAND_KEY,
						  message,
						  this);
						  
	message = new BMessage(INVOKE_BUTTON);
	message->AddString("which", SET_BUTTON_TITLE);
	Window()->AddShortcut('S', B_COMMAND_KEY,
						  message,
						  this);
}


/* ...............................................................	
	WindowColorsPanel::AllAttached()
	Seemed like the best place to set the focus view after all
	of the dust has settled.
   ............................................................... */

void WindowColorsPanel::AllAttached()
{
	_windowView->ChildAt(0)->MakeFocus(true);
}


/* ...............................................................	
	WindowColorsPanel::Draw()
	Fills in the update rectangle in background gray, and draws
	the beveled appearance for the base view.
   ............................................................... */

void WindowColorsPanel::Draw(BRect updateRect)
{
	/* Get the bounds, and calculate an inset
	   by (1, 1) background fill region intersection
	   with the updateRect.  Fill it in "standard" gray. */
	BRect bounds = Bounds();
	BRect backgroundRect = bounds;
	backgroundRect.InsetBy(1, 1);
	
	/* Update the background if necessary */
	if (backgroundRect.Intersects(updateRect))
	{
		SetLowColor(RGB_COLOR_216);
		FillRect((backgroundRect & updateRect), B_SOLID_LOW);
	}
	
	/* Draw the beveled background (no matter what) */
	BeginLineArray(4);
	AddLine(BPoint(bounds.left, bounds.top),
			BPoint(bounds.left, bounds.bottom),
			WHITE_COLOR);
	AddLine(BPoint(bounds.left + 1, bounds.top),
			BPoint(bounds.right, bounds.top),
			WHITE_COLOR);
	AddLine(BPoint(bounds.right, bounds.top + 1),
			BPoint(bounds.right, bounds.bottom),
			RGB_COLOR_192);
	AddLine(BPoint(bounds.right - 1, bounds.bottom),
			BPoint(bounds.left + 1, bounds.bottom),
			RGB_COLOR_192);
	EndLineArray();
}


/* ...............................................................
	WindowColorsPanel::Show()
	Hands the default status back to the "Set Colors" button.
   ............................................................... */
		
void WindowColorsPanel::Show()
{
	_setButton->MakeDefault(true);
	BView::Show();
}


/* ...............................................................	
	WindowColorsPanel::MessageReceived()
	Called when the view receives a message
   ............................................................... */

void WindowColorsPanel::MessageReceived(BMessage *message)
{
	switch (message->what)
	{			
		/* Invokes a button (via key shortcut) */
		case INVOKE_BUTTON:
		{
			const char *name;
			if (message->FindString("which", &name) == B_OK)
			{
				BButton *button = (BButton *)FindView(name);
				if (button)
				{
					button->MouseDown(BPoint(5, 5));
					button->Invoke();
				}
			}
		}
		break;			
			
		/* Sets all of the window colors */
		case SET_WINDOW_COLORS:
			if (_setButton->IsEnabled())
				SetSystemWindowColors(true);
			break;
			
		/* Reverts the color model to the current colors */
		case REVERT_WINDOW_COLORS:
			if (_revertButton->IsEnabled())
				GetSystemWindowColors(true);
			break;
			
		/* Reverts to the exact BeOS defaults */
		case REVERT_TO_BE_DEFAULTS:
			if (_defaultsButton->IsEnabled())
				GetDefaultWindowColors(true);
			break;
			
		/* Switches to the AboutPanel */
		case B_ABOUT_REQUESTED:
			if (! IsHidden())
				((WindowColorsWindow *)Window())->
					ShowAboutPanel(true);
			break;
			
		/* Pass it on... */
		default:
			BView::MessageReceived(message);
			break;
	}
}


/* ...............................................................	
	WindowColorsPanel::SetColor()
	Sets one of the window_color colors in the color model to
	the specified color.  Could be cleaned up.
   ............................................................... */

void WindowColorsPanel::SetColor(window_color which, 
								 rgb_color color)
{
	_colors[which] = color;
	
	int32 updateView = 0;	/* _windowView */
	switch (which)
	{	   
		case ACTIVE_TAB_BASE_COLOR:
		case INACTIVE_TAB_BASE_COLOR:
		case ACTIVE_FRAME_BASE_COLOR:
		case INACTIVE_FRAME_BASE_COLOR:
		{
			_colors[which - 1] = determine_hilite_color(which, color);
			_colors[which + 1] = determine_shadow_color(which, color);
			
			if (which == INACTIVE_FRAME_BASE_COLOR)
				_colors[RESIZE_CORNER_COLOR] = color;
			
			bool getDefaults;
			if (which == ACTIVE_TAB_BASE_COLOR)
			{
				getDefaults = compare_colors(DEFAULT_WINDOW_COLORS
										    [ACTIVE_TAB_BASE_COLOR],
										    color);
				if (getDefaults)
					get_window_bitmap(_bitmaps[CLOSE_BOX_OFF_BITMAP],
									  CLOSE_BOX_OFF_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(_bitmaps[CLOSE_BOX_OFF_BITMAP], 
										CLOSE_BOX_OFF_BITMAP, color);
				
				if (getDefaults)
					get_window_bitmap(_bitmaps[CLOSE_BOX_ON_BITMAP], 
									  CLOSE_BOX_ON_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(_bitmaps[CLOSE_BOX_ON_BITMAP], 
										CLOSE_BOX_ON_BITMAP, color);
										
				if (getDefaults)
					get_window_bitmap(_bitmaps[ZOOM_BOX_OFF_BITMAP], 
									  ZOOM_BOX_OFF_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(_bitmaps[ZOOM_BOX_OFF_BITMAP], 
										ZOOM_BOX_OFF_BITMAP, color);

				if (getDefaults)
					get_window_bitmap(_bitmaps[ZOOM_BOX_ON_BITMAP], 
									  ZOOM_BOX_ON_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(_bitmaps[ZOOM_BOX_ON_BITMAP], 
										ZOOM_BOX_ON_BITMAP, color);
			}
		}
		break;
					   
		case ACTIVE_MODAL_FRAME_BASE_COLOR:
		case INACTIVE_MODAL_FRAME_BASE_COLOR:
		{
			updateView = 1;		/* _modalWindowView */
			_colors[which - 1] = determine_hilite_color(which, color);
			_colors[which + 1] = determine_shadow_color(which, color);
		}
		break;
		   			
		case BORDER_COLOR:
		{
			updateView = 2;		/* both */
			_colors[BORDER_SHADOW_COLOR] = determine_shadow_color(which,
																  color);
			_colors[RESIZE_CORNER_HILITE_COLOR] = color;
		}
		break;
	}
			
	/* Invalidate the correct view */
	if (! IsHidden())
	{
		if (updateView == 2)
		{
			_windowView->Draw(_windowView->Bounds());
			_modalWindowView->Draw(_modalWindowView->Bounds());
		}
		else
			ChildAt(updateView)->Draw(ChildAt(updateView)->Bounds());

		if (_synced)
		{
			_synced = false;
			_setButton->SetEnabled(true);
			_setButton->SetFlags(B_WILL_DRAW); /* grr */
			_revertButton->SetEnabled(true);
			_revertButton->SetFlags(B_WILL_DRAW);
		}
	}
}


/* ...............................................................	
	WindowColorsPanel::Bitmap()
	Returns the specified bitmap in the model.
   ............................................................... */

BBitmap *WindowColorsPanel::Bitmap(window_bitmap which)
{
	return _bitmaps[which];
}


/* ...............................................................	
	WindowColorsPanel::Color()
	Returns the specified window_color in the color model used
	by the WindowViews for drawing.
   ............................................................... */

rgb_color WindowColorsPanel::Color(window_color which)
{
	return _colors[which];
}


/* ...............................................................	
	WindowColorsPanel::GetSystemWindowColors()
	Spins through all of the window_color entries in the system,
	and sets the corresponding colors in the color model used by
	the WindowViews.  If refresh is true, it redraws all of the
	WindowViews since all colors are affected.
   ............................................................... */

void WindowColorsPanel::GetSystemWindowColors(bool refresh)
{
	/* Get the active window colors into our model array */
	get_window_colors(_colors, ACTIVE_ONES);
	
	/* Get the active window bitmaps too */		
	for (uint8 i = 0; i < 4; i++)
	{
		get_window_bitmap(_bitmaps[i],
						 (window_bitmap)i);
	}
	
	/* Refresh all of the WindowViews if necessary */
	if (refresh)
	{
		_windowView->Update();
		_modalWindowView->Update();
		
		if (_setButton != NULL)
			_setButton->SetEnabled(FALSE);
		if (_revertButton != NULL)
			_revertButton->SetEnabled(FALSE);
		if (_colorPicker != NULL)
			_colorPicker->SetValueQuietly(FocusedColorLabel()->
										   Color());
	}
	
	/* We're synced with the system now */
	_synced = true;
}


/* ...............................................................	
	WindowColorsPanel::GetDefaultWindowColors()
	Fills our color model with the default BeOS window colors.
   ............................................................... */

void WindowColorsPanel::GetDefaultWindowColors(bool refresh)
{
	get_window_colors(_colors, DEFAULT_ONES);
	
	/* Get the default window bitmaps too */		
	for (uint8 i = 0; i < 4; i++)
		get_window_bitmap(_bitmaps[i], (window_bitmap)i,
						  DEFAULT_ONES);
	
	if (refresh)
	{
		/* Redraw the window views and children */
		_windowView->Update();
		_modalWindowView->Update();
		
		/* Update the color control value */
		if (_colorPicker != NULL)
			_colorPicker->SetValueQuietly(FocusedColorLabel()->
										  Color());
		/* Set up the buttons */
		if (_setButton)
		{
			_setButton->SetEnabled(true);
			_setButton->SetFlags(B_WILL_DRAW);
		}
		if (_revertButton)
		{
			_revertButton->SetEnabled(true);
			_revertButton->SetFlags(B_WILL_DRAW);
		}
	}
	_synced = false;
}


/* ...............................................................	
	WindowColorsPanel::SetSystemWindowColors()
	Sets each of the system window colors to the respective colors 
	in the model.
   ............................................................... */

void WindowColorsPanel::SetSystemWindowColors(bool refresh)
{		
	/* Set the window colors from our color model */
	set_window_colors(_colors, refresh);
	
	/* Save our color model set as the preferred one */
	window_colors_app->SavePreferredSet(&_colors[0]);
	
	/* I'll fix this when I "learn how to program" ;) */
	_synced = true;
	
	/* To only use these as the mechanism */
	_setButton->SetEnabled(false);
	_revertButton->SetEnabled(false);
}


/* ...............................................................	
	ActiveWindowView
	Returns the active WindowView subview that should be drawn in its
	"active state," all other windows being in an "inactive" state.
   ............................................................... */

WindowView *WindowColorsPanel::ActiveWindowView()
{
	return _activeWindowView;
}


/* ...............................................................	
	GetSystemWindowColors
	Initializes the color model array to match the colors used for 
	windows in the system.
   ............................................................... */

void WindowColorsPanel::SetActiveWindowView(WindowView *view)
{
	if (view != NULL)
	{
		if (_activeWindowView != NULL)
			_activeWindowView->Activate(false);
		_activeWindowView = view;
		view->Activate(true);
	}
}


/* ...............................................................
	FocusedColorLabel
	Returns the color label with the current focus.
   ............................................................... */

ColorLabel *WindowColorsPanel::FocusedColorLabel()
{
	return ((ColorLabel *)Window()->CurrentFocus());
}


/* ...............................................................
	ColumnLabelFilter constructor
	Constructs a ColumnLabelFilter object.
   ............................................................... */

ColumnLabelFilter::ColumnLabelFilter()
				  :BMessageFilter(B_ANY_DELIVERY, 
								  B_ANY_SOURCE,
								  B_MOUSE_DOWN)
{
}


/* ...............................................................
	ColumnLabelFilter::Filter()
	Filters B_MOUSE_DOWN objects from one of our BStringView
	column labels, and sets the focus to the first entry in
	the correct column, ignoring the message.
   ............................................................... */

filter_result ColumnLabelFilter::Filter(BMessage *message,
									    BHandler **target)
{
#pragma unused(message)
	BView *viewTarget = ((BView *)*target);
	BView *parent = viewTarget->Parent();
	
	if (parent != NULL)
	{
		/* Forget the message.  We'll set the focus of
		   the appropriate column's first ColorLabel. */
		int32 chosenOne = 5;
		
		if (strcmp(viewTarget->Name(), 
			       TITLE_TAB_LABEL_TEXT) == 0)
				chosenOne = 0;
				
		else if (strcmp(viewTarget->Name(), 
				 	    FRAME_LABEL_TEXT) == 0)
				chosenOne = 3;
				
		else if (strcmp(viewTarget->Name(), 
					    MODAL_FRAME_LABEL_TEXT) == 0)
				chosenOne = 0;
				
		if (((WindowView *)parent)->IsActive() == false &&
			modifiers() & B_OPTION_KEY)
				chosenOne++;
				
		if (chosenOne < 5)
			parent->ChildAt(chosenOne)->MakeFocus(true);
	}
	return B_SKIP_MESSAGE;
}